<?php

if (!defined('GLPI_ROOT')) {
   die("Sorry. You can't access directly to this file");
}

/**
 * Class PluginTRT16Profile
 *
 * This class manages the profile rights of the plugin
 */
class PluginTRT16Profile extends Profile {

   /**
    * Get tab name for item
    *
    * @param CommonGLPI $item
    * @param type       $withtemplate
    *
    * @return string
    */
   function getTabNameForItem(CommonGLPI $item, $withtemplate = 0) {

      if ($item->getType() == 'Profile'
          && $item->getField('interface') != 'helpdesk') {
         return __('TRT16', 'trt16');
      }
      return '';
   }

   /**
    * display tab content for item
    *
    * @global type      $CFG_GLPI
    *
    * @param CommonGLPI $item
    * @param type       $tabnum
    * @param type       $withtemplate
    *
    * @return boolean
    */
   static function displayTabContentForItem(CommonGLPI $item, $tabnum = 1, $withtemplate = 0) {
      global $CFG_GLPI;

      if ($item->getType() == 'Profile') {
         $ID   = $item->getID();
         $prof = new self();

         self::addDefaultProfileInfos($ID,
                                      ['plugin_trt16' => 0]);
         $prof->showForm($ID);
      }

      return true;
   }

   /**
    * show profile form
    *
    * @param type $ID
    * @param type $options
    *
    * @return boolean
    */
   function showForm($profiles_id = 0, $openform = TRUE, $closeform = TRUE) {

      echo "<div class='firstbloc'>";
      if (($canedit = Session::haveRightsOr(self::$rightname, [CREATE, UPDATE, PURGE]))
          && $openform) {
         $profile = new Profile();
         echo "<form method='post' action='" . $profile->getFormURL() . "'>";
      }

      $profile = new Profile();
      $profile->getFromDB($profiles_id);

      $rights = $this->getAllRights();
      $profile->displayRightsChoiceMatrix($rights, ['canedit'       => $canedit,
                                                         'default_class' => 'tab_bg_2',
                                                         'title'         => __('General')]);
      if ($canedit
          && $closeform) {
         echo "<div class='center'>";
         echo Html::hidden('id', ['value' => $profiles_id]);
         echo Html::submit(_sx('button', 'Save'), ['name' => 'update', 'class' => 'btn btn-primary']);
         echo "</div>\n";
         Html::closeForm();
      }
      echo "</div>";

      $this->showLegend();
   }

   /**
    * Get all rights
    *
    * @param type $all
    *
    * @return array
    */
   static function getAllRights($all = false) {

      $rights = [
         ['itemtype' => 'PluginTRT16Group',
               'label'    => __('TRT16', 'trt16'),
               'field'    => 'plugin_trt16'
         ]
      ];

      return $rights;
   }

   /**
    * Init profiles
    *
    **/

   static function translateARight($old_right) {
      switch ($old_right) {
         case '':
            return 0;
         case 'r' :
            return READ;
         case 'w':
            return ALLSTANDARDRIGHT;
         case '0':
         case '1':
            return $old_right;

         default :
            return 0;
      }
   }


   /**
    * @since 0.85
    * Migration rights from old system to the new one for one profile
    *
    * @param $profiles_id the profile ID
    */
   static function migrateOneProfile($profiles_id) {
      global $DB;
      //Cannot launch migration if there's nothing to migrate...
      if (!$DB->tableExists('glpi_plugin_trt16_profiles')) {
         return true;
      }

      foreach ($DB->request('glpi_plugin_trt16_profiles',
                            "`profiles_id`='$profiles_id'") as $profile_data) {

         $matching       = ['show_trt16_tab' => 'plugin_trt16'];
         $current_rights = ProfileRight::getProfileRights($profiles_id, array_values($matching));
         foreach ($matching as $old => $new) {
            if (!isset($current_rights[$old])) {
               $right = self::translateARight($profile_data[$old]);
               switch ($new) {
                  case 'plugin_trt16' :
                     $right = 0;
                     if ($profile_data[$old] == '1') {
                        $right = ALLSTANDARDRIGHT;
                     }
                     break;
               }

               $query = "UPDATE `glpi_profilerights` 
                         SET `rights`='" . $right . "' 
                         WHERE `name`='$new' AND `profiles_id`='$profiles_id'";
               $DB->query($query);
            }
         }
      }
   }

   static function initProfile() {
      global $DB;

      $profile = new self();
      $dbu     = new DbUtils();

      //Add new rights in glpi_profilerights table
      foreach ($profile->getAllRights() as $data) {
         if ($dbu->countElementsInTable("glpi_profilerights",
                                        ['name' => $data['field']]) == 0) {
            ProfileRight::addProfileRights([$data['field']]);
         }
      }

      foreach ($DB->request('glpi_profilerights',
                            ['profiles_id' => $_SESSION['glpiactiveprofile']['id'],
                             'name'        => ['LIKE', '%plugin_trt16%']]) as $prof) {
         $_SESSION['glpiactiveprofile'][$prof['name']] = $prof['rights'];
      }
   }

   /**
    * Initialize profiles, and migrate it necessary
    */
   static function changeProfile() {
      global $DB;

      foreach ($DB->request("SELECT *
                           FROM `glpi_profilerights` 
                           WHERE `profiles_id`='" . $_SESSION['glpiactiveprofile']['id'] . "' 
                              AND `name` LIKE '%plugin_trt16%'") as $prof) {
         $_SESSION['glpiactiveprofile'][$prof['name']] = $prof['rights'];
      }

   }

   static function createFirstAccess($profiles_id) {

      $rights = ['plugin_trt16' => ALLSTANDARDRIGHT];

      self::addDefaultProfileInfos($profiles_id,
                                   $rights, true);

   }

   /**
    * @param $profile
    **/
   static function addDefaultProfileInfos($profiles_id, $rights, $drop_existing = false) {

      $profileRight = new ProfileRight();
      $dbu = new DbUtils();
      foreach ($rights as $right => $value) {
         if ($dbu->countElementsInTable('glpi_profilerights',
                                  ["profiles_id" => $profiles_id,
                                   "name"        => $right]) && $drop_existing) {
            $profileRight->deleteByCriteria(['profiles_id' => $profiles_id, 'name' => $right]);
         }
         if (!$dbu->countElementsInTable('glpi_profilerights',
                                   ["profiles_id" => $profiles_id,
                                    "name"        => $right])) {
            $myright['profiles_id'] = $profiles_id;
            $myright['name']        = $right;
            $myright['rights']      = $value;
            $profileRight->add($myright);

            //Add right to the current session
            $_SESSION['glpiactiveprofile'][$right] = $value;
         }
      }
   }

   static function removeRightsFromSession() {
      foreach (self::getAllRights(true) as $right) {
         if (isset($_SESSION['glpiactiveprofile'][$right['field']])) {
            unset($_SESSION['glpiactiveprofile'][$right['field']]);
         }
      }
   }

   static function install(Migration $mig) {
      global $DB;

      $table = 'glpi_plugin_trt16_profiles';
      if (!$DB->tableExists($table)
          && !$DB->tableExists('glpi_plugin_trt16_preferences')) {

         self::initProfile();
         self::createFirstAccess($_SESSION['glpiactiveprofile']['id']);

      } else {
         if ($DB->tableExists($table)
             && $DB->fieldExists($table,'ID')) { //< 0.7.0
            $mig->changeField($table, 'ID', 'id', 'autoincrement');
         }
         // -- SINCE 0.85 --
         //Add new rights in glpi_profilerights table
         $profileRight = new ProfileRight();

         if ($DB->tableExists($table)) {
            foreach ($DB->request($table, ['use' => 1]) as $data) {
               $right['profiles_id']   = $data['id'];
               $right['name']          = "plugin_trt16";
               $right['rights']        = $data['use'];

               $profileRight->add($right);
            }
            $mig->dropTable($table);
         }
      }
   }

}